#!/bin/bash

set -e

SERVICES="stcapi-rh stcapi-fcgi stcapi-sl stcweb testcenter-server"

# Get all running container IDs
CONTAINER_IDS=$(docker ps -q)

if [ -z "$CONTAINER_IDS" ]; then
    echo "No running containers found."
    exit 0
fi

# Process each container
for CONTAINER_ID in $CONTAINER_IDS; do
    echo "Checking container $CONTAINER_ID..."

    # Process only labserver containers.
    if docker exec -it "$CONTAINER_ID" sh -c "ls /home/testcenter/server/stcbll.ini" > /dev/null 2>&1; then
        echo "Container $CONTAINER_ID is a labserver container. Processing..."

        STOPPED_SERVICES=""

        # Copy files to container
        docker cp . "$CONTAINER_ID:/home/testcenter/server"

        # Stop services (only if they exist and are running)
        for service in $SERVICES; do
            if docker exec -it "$CONTAINER_ID" sh -c "supervisorctl status $service 2>/dev/null | grep -q RUNNING"; then
                echo "Stopping $service..."
                docker exec -it "$CONTAINER_ID" sh -c "supervisorctl stop $service"
                STOPPED_SERVICES="$STOPPED_SERVICES $service"
            fi
        done

        # Run the patch tool
        docker exec -it "$CONTAINER_ID" sh -c "cd /home/testcenter/server && ./bin/tc-patch-2025 ."

        # Set the owner/group to testcenter:testcenter
        docker exec -it "$CONTAINER_ID" sh -c "chown -R testcenter:testcenter /home/testcenter/server/libcore.so" || true

        # Start services that were stopped (in reverse order)
        for service in $(echo $STOPPED_SERVICES | awk '{for(i=NF;i>=1;i--) printf "%s ", $i}'); do
            echo "Starting $service..."
            OUTPUT=$(docker exec -it "$CONTAINER_ID" sh -c "supervisorctl start $service" 2>&1)
            EXIT_CODE=$?
            if echo "$OUTPUT" | grep -q "already started"; then
                # Suppress "already started" messages
                :
            else
                echo "$OUTPUT"
                if [ $EXIT_CODE -ne 0 ]; then
                    exit 1
                fi
            fi
        done

        echo "Finished processing container $CONTAINER_ID"
    else
        echo "Container $CONTAINER_ID is not a labserver container. Skipping..."
    fi

    echo ""
done

echo "All labserver containers processed."
exit 0
